package org.biopax.paxtools.io.sbgn;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;

import org.ivis.layout.*;
import org.ivis.layout.util.*;
import org.ivis.layout.cose.CoSELayout;
import org.sbgn.bindings.Arc;
import org.sbgn.bindings.Port;
import org.sbgn.bindings.Glyph;
import org.sbgn.bindings.Sbgn;
import org.sbgn.bindings.Bbox;

/**
 * Class for applying layout by ChiLay component to Sbgn generated by paxtools.
 * @author: istemi Bahceci
 * */

public class SBGNLayoutManager
{
	
	private Layout layout;
	private VCompound root;
	
	// mapping between view and layout level 
	private HashMap <VNode, LNode> viewToLayout;
	private HashMap <LNode, VNode> layoutToView;
	private HashMap <Glyph,VNode>  glyphToVNode;
	private HashMap <String, Glyph> idToGLyph;
	private HashMap<String, Glyph> idToCompartmentGlyphs;
	private HashMap<String, Glyph> portIDToOwnerGlyph;
	
	
	/**
	 * Applies CoSE layout to the given SBGN model.
	 * 
	 * @param sbgn Sbgn object to which layout will be applied
	 * @return Laid out sbgn object
	 */
	public Sbgn createLayout(Sbgn sbgn)
	{
		viewToLayout = new HashMap();
		layoutToView = new HashMap();
		glyphToVNode = new HashMap();
		idToGLyph = new HashMap();
		idToCompartmentGlyphs = new HashMap();
		portIDToOwnerGlyph = new HashMap();
		
		// Using Compound spring  embedder layout
		this.layout = new CoSELayout();
		
		LGraphManager graphMgr = this.layout.getGraphManager(); 
		LGraph lRoot = graphMgr.addRoot();
		this.root = new VCompound(new Glyph());
		//lRoot.vGraphObject = this.root;
		
		
		for (Glyph g: sbgn.getMap().getGlyph()) 
		{
			if(g.getClazz() == "compartment")
			{
				idToCompartmentGlyphs.put(g.getId(), g);
			}			
		}
		
		ArrayList <Glyph> deletedList = new ArrayList<Glyph>();
		for (Glyph g: sbgn.getMap().getGlyph()) 
		{	
			if(g.getCompartmentRef() != null)
			{
				Glyph containerCompartment = (Glyph)g.getCompartmentRef();
				idToCompartmentGlyphs.get(containerCompartment.getId()).getGlyph().add(g);
				deletedList.add(g);	
			}	
		}
		
		for (Glyph g: deletedList) 
		{	
			sbgn.getMap().getGlyph().remove(g);
		}
					
		// Create Vnodes for ChiLay layout component
		createVNodes(root, sbgn.getMap().getGlyph());
		
		for (VNode vNode: this.root.children) 
		{ 
			this.createLNode(vNode, null, this.layout); 
		}
		
		// Create LEdges for ChiLay layout component
		createLEdges(sbgn.getMap().getArc(), this.layout);
		
		graphMgr.updateBounds();
		
		// Apply layout
		this.layout.runLayout();
		
		graphMgr.updateBounds();

		for (VNode vNode: this.root.children) 
		{ 
			updateCompoundBounds(vNode.glyph, vNode.glyph.getGlyph()); 
		}
		
		/*for (Glyph compGlyph: idToCompartmentGlyphs.values()) 
		{
			compGlyph.getGlyph().clear();
		}*/

		return sbgn;
	}
	
	
	public void updateCompoundBounds(Glyph parent,List<Glyph> childGlyphs)
	{		
		float PAD = (float) 2.0;
		float minX = Float.MAX_VALUE; float minY = Float.MAX_VALUE;
		float maxX = Float.MIN_VALUE; float maxY = Float.MIN_VALUE;
		
		for (Glyph tmpGlyph:childGlyphs) 
		{
			if(tmpGlyph.getClazz() != "unit of information" && tmpGlyph.getClazz() != "state variable" )
			{
				if(tmpGlyph.getGlyph().size() > 0)
					updateCompoundBounds(tmpGlyph, tmpGlyph.getGlyph());
				
	            float w = tmpGlyph.getBbox().getW();
				float h = tmpGlyph.getBbox().getH();
				
	            // Verify MIN and MAX x/y again:
	            minX = Math.min(minX, (tmpGlyph.getBbox().getX()));
	            minY = Math.min(minY, (tmpGlyph.getBbox().getY()));
	            maxX = Math.max(maxX, (tmpGlyph.getBbox().getX())+w);
	            maxY = Math.max(maxY, (tmpGlyph.getBbox().getY())+h);
	            
	            if (minX == Float.MAX_VALUE) minX = 0;
	            if (minY == Float.MAX_VALUE) minY = 0;
	            if (maxX == Float.MIN_VALUE) maxX = 0;
	            if (maxY == Float.MIN_VALUE) maxY = 0;
	            
	            parent.getBbox().setX(minX - PAD);
	            parent.getBbox().setY(minY - PAD);
	            parent.getBbox().setW(maxX -  parent.getBbox().getX() + PAD);
	            parent.getBbox().setH(maxY -  parent.getBbox().getY() + PAD);
			}
		}
	}
	
	public boolean isChildless(Glyph tmpGlyph)
	{
		boolean checker = true;
		for(Glyph glyph: tmpGlyph.getGlyph() )
		{
			if (glyph.getClazz() !=  "state variable" && glyph.getClazz() !=  "unit of information"  ) 
			{
				checker = false;
				break;
			}
		}
		return checker;
	}
	
	/**
	 * Recursively creates VNodes from Glyphs of Sbgn. 
	 * 
	 * @param parent Parent of the glyphs that are passed as second arguement.
	 * @param glyphs Glyphs that are child of parent which is passed as first arguement.
	 * 
	 * */
	public void createVNodes(VCompound parent,List<Glyph> glyphs)
	{
		for(Glyph glyph: glyphs )
		{	
			if (glyph.getClazz() !=  "state variable" && glyph.getClazz() !=  "unit of information"  ) 
			{			
				if(!this.isChildless(glyph))
				{
					VCompound v = new VCompound(glyph);

					idToGLyph.put(glyph.getId(), glyph);
					glyphToVNode.put(glyph, v);
					parent.children.add(v);
					createVNodes(v, glyph.getGlyph());
				}
				
				else
				{
					VNode v = new VNode(glyph);
					idToGLyph.put(glyph.getId(), glyph);
					glyphToVNode.put(glyph, v);		
					parent.children.add(v);
				}
				
				for(Port p: glyph.getPort())
				{
					portIDToOwnerGlyph.put(p.getId(), glyph );
				}
				
			}
		}
	}
	
	/**
	 *  Briefly, finds the LNode which is parent of the VNode that 
	 *  corresponds to the glyph object which is container of this port.
	 * 
	 * @param p port object that by its ID  the corresponding LNode will be retrieved.
	 * @return  LNode which is parent of the VNode that corresponds to the glyph object which is container of this port p.
	 * 
	 * */
	public LNode getLNodeByPort(Port p)
	{
		String tmpStr = p.getId();
		VNode tmpVNode = glyphToVNode.get(portIDToOwnerGlyph.get(tmpStr));
		// Return corresponding LNode
		return viewToLayout.get(tmpVNode);
	}
	
	/**
	 * Creates LNodes from Arcs of Sbgn and adds it to the passed layout object. 
	 * 
	 * @param arcs List of arc objects from which the LEdges will be constructed for ChiLay Layout component.
	 * @param layout layout object to which the created LEdges added.
	 * 
	 * */
	public void createLEdges(List<Arc> arcs, Layout layout)
	{
		for(Arc arc: arcs )
		{
			LEdge lEdge = layout.newEdge(null); 
			LNode sourceLNode;
			LNode targetLNode;
			
			// If source is port, first clear port indicators else retrieve it from hashmaps
			if (arc.getSource() instanceof Port ) 
			{
				sourceLNode = getLNodeByPort((Port)arc.getSource());
				arc.setSource(layoutToView.get(sourceLNode).glyph);
			}
			else
			{
				sourceLNode = this.viewToLayout.get(glyphToVNode.get(arc.getSource()));
			}
			
			
			// If target is port, first clear port indicators else retrieve it from hashmaps
			if (arc.getTarget() instanceof Port) 
			{
				targetLNode = getLNodeByPort((Port)arc.getTarget());
				arc.setTarget(layoutToView.get(targetLNode).glyph);
			}
			else
			{
				targetLNode = this.viewToLayout.get(glyphToVNode.get(arc.getTarget()));
			}
			
	
			// Add edge to the layout
			this.layout.getGraphManager().add(lEdge, sourceLNode, targetLNode);
		}
	}
	
	
	/**
	 * Helper function for creating LNode objects from VNode objects and adds them to the given layout.
	 * 
	 * @param vNode  VNode object from which a corresponding LNode object will be created.
	 * @param parent parent of vNode, if not null vNode will be added to layout as child node.
	 * @param layout layout object to which the created LNodes added.
	 * */
	
	public void createLNode(VNode vNode,VNode parent,Layout layout)
	{
		LNode lNode = layout.newNode(vNode); 

		LGraph rootLGraph = layout.getGraphManager().getRoot();
		
		this.viewToLayout.put(vNode, lNode); 
		this.layoutToView.put(lNode, vNode);
		
		// if the vNode has a parent, add the lNode as a child of the parent l-node. 
		// otherwise, add the node to the root graph. 
		if (parent != null) 
		{ 
			LNode parentLNode = this.viewToLayout.get(parent); 
			parentLNode.getChild().add(lNode); 
		} 
		else 
		{ 
			rootLGraph.add(lNode); 
		}		
		
		lNode.setLocation(vNode.glyph.getBbox().getX(),vNode.glyph.getBbox().getY());
			
		if (vNode instanceof VCompound) 
		{ 
			VCompound vCompound = (VCompound) vNode; 
			// add new LGraph to the graph manager for the compound node 
			layout.getGraphManager().add(layout.newGraph(null), lNode); 
			// for each VNode in the node set create an LNode 
			for (VNode vChildNode: vCompound.getChildren()) 
			{ 
				this.createLNode(vChildNode, vCompound, layout); 
				
			} 		
		}
		else
		{
			lNode.setWidth(vNode.glyph.getBbox().getW());
			lNode.setHeight(vNode.glyph.getBbox().getH());
		}
	}
}